# python
##########################################################################
#
# scriptshelf.py
#
# Version: 0.32
#
# Author: Bernd C. Moeller
#
# Description: This kit adds a script shelf to modo. You can simply drag and drop your scripts on there, it will save the script and will generate a button.
#
# Last Update: 2015/09/15
#
##########################################################################

import lx
import lxifc
import lxu.command
from PySide.QtCore import *
from PySide.QtGui import *
import os
import traceback
import string
import shutil
import webbrowser

KITPATH = lx.eval('query platformservice alias ? {kit_scriptShelf:}')

#KITPATH = FILEPATH = 'C:/Users/bernd/AppData/Roaming/Luxology/Kits/scriptShelf/'
DATAPATH = os.path.join(KITPATH, 'data')
SCRIPTSPATH = os.path.join(DATAPATH, 'pythonscripts')
ICONSPATH = os.path.join(DATAPATH, 'icons')
QWIDGETSIZE_MAX = 16777215

def format_filename(s):
    valid_chars = "-_() %s%s" % (string.ascii_letters, string.digits)
    filename = ''.join(c for c in s if c in valid_chars)
    filename = filename.replace(' ', '_')  # I don't like spaces in filenames.
    return filename

# check file exists dialog box thingy, might not be the best solution
# maybe it's better to use modo dialogs for consistency (keyboard
# shortcuts, look)
def fileDialogBox(parent, title, label, files, filename = ""):
    #files = [f[:-3] for f in os.listdir(SCRIPTSPATH) if f.endswith(".py")]
    a, ok = QInputDialog.getText(parent, title, label, text=format_filename(filename))
    a = format_filename(a)
    # here should be a check for a valid file name
    if ok:
        if a in files:
            msgBox = QMessageBox(parent)
            msgBox.setText('File already exists. Replace?')
            msgBox.setStandardButtons(QMessageBox.Yes | QMessageBox.No)
            ret = msgBox.exec_()
            if ret == QMessageBox.Yes:
                return (a, True)
            else:
                return fileDialogBox(parent, title, label, files)
        elif not a:
            return fileDialogBox(parent, title, label, files)
        else:
            return (a, False)
    else:
        return (None, False)

class RightClickScriptButton(QToolButton):

    def __init__(self, name, parent=None):
        # super stuff
        super(RightClickScriptButton, self).__init__()
        self.setText(name)
        self.setStyleSheet("""QToolButton {text-align: left;
        padding-top: 2px;
        padding-bottom: 2px;
        padding-left: 7px;
        padding-right: 7px;}
        QToolButton::menu-indicator { image: none; }""")
        if name == 'HELP':
            self.setStyleSheet(self.styleSheet() + 'QToolButton {font-weight:bold;}')

        self.setToolButtonStyle(parent.iconDisp)
        self.setSizePolicy(QSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed))

        #self.setIcon(self.style().standardIcon(QStyle.SP_DirIcon))

        # get supported fileextensions
        fExt = [str(f) for f in QImageReader.supportedImageFormats()]
        # read icons in icondir
        icons = [f for f in os.listdir(ICONSPATH) if os.path.splitext(f.lower())[1][1:] in fExt]
        iconsNames = [os.path.splitext(f)[0] for f in icons]

        if name in iconsNames:
            ind = iconsNames.index(name)
            self.setIcon(QIcon(os.path.join(ICONSPATH,icons[ind])))
        else:
            self.setIcon(QIcon(os.path.join(KITPATH,'scriptShelfIcon_32.png')))
        self.setIconSize(QSize(32,32))

        # setup for right click menu
        self.setContextMenuPolicy(Qt.ActionsContextMenu)
        # create and add rename menu item
        rename = QAction(self)
        rename.setText('Rename Button')
        rename.triggered.connect(self.renameButton)
        self.addAction(rename)
        # create and add remove menu item
        remove = QAction(self)
        remove.setText("Remove Button")
        remove.triggered.connect(self.removeButton)
        self.addAction(remove)

        #separator
        sep = QAction(self)
        sep.setSeparator(True)
        self.addAction(sep)

        # create and add edit menu item
        edit = QAction(self)
        edit.setText('Edit Script')
        edit.triggered.connect(self.editButton)
        self.addAction(edit)
        # create and add delete menu item
        delete = QAction(self)
        delete.setText("Delete Script")
        delete.triggered.connect(lambda : self.removeButton(True))
        self.addAction(delete)

        #separator
        sep2 = QAction(self)
        sep2.setSeparator(True)
        self.addAction(sep2)

        # create and add delete menu item
        delete = QAction(self)
        delete.setText("Change Icon")
        delete.triggered.connect(self.changeIcon)
        self.addAction(delete)
        # create and add delete menu item
        delete = QAction(self)
        delete.setText("Remove Icon")
        delete.triggered.connect(self.removeIcon)
        self.addAction(delete)

        # set command that's executed when button is clicked
        self.clicked.connect(self.runScript)

    def changeIcon(self, fileName = None):
        # pop up file dialog ###DUP###
        fExt = [str(f) for f in QImageReader.supportedImageFormats()]
        # read icons in icondir
        icons = [f for f in os.listdir(ICONSPATH) if os.path.splitext(f.lower())[1][1:] in fExt]
        iconsNames = [os.path.splitext(f)[0] for f in icons]
        print icons, iconsNames

        files = 'Icon Files (%s)'%' *.'.join(fExt)
        if not fileName:
            fileName = QFileDialog.getOpenFileName(None, "Open Image", "", files)[0]
        # check if file and ok
        if fileName:
            # delete old icon if there
            name = self.text()
            if name in iconsNames:
                ind = iconsNames.index(name)
                os.remove(os.path.join(ICONSPATH, icons[ind]))
            # copy icon
            newPath = os.path.join(ICONSPATH, '%s%s'%(name, os.path.splitext(fileName)[1]))
            shutil.copyfile(fileName, newPath)
            # change icon
            self.setIcon(QIcon(newPath))
            self.parent().initAll()

    def removeIcon(self):
        # set default icon
        self.setIcon(QIcon(os.path.join(KITPATH,'scriptShelfIcon.png')))
        # delete icon file ###DUP###
        fExt = [str(f) for f in QImageReader.supportedImageFormats()]
        # read icons in icondir
        icons = [f for f in os.listdir(ICONSPATH) if os.path.splitext(f.lower())[1][1:] in fExt]
        iconsNames = [os.path.splitext(f)[0] for f in icons]
        name = self.text()
        if name in iconsNames:
            ind = iconsNames.index(name)
            os.remove(os.path.join(ICONSPATH, icons[ind]))
            self.parent().initAll()

    def removeButton(self, deleteFile = False):
        # check if the user is sure about this decision
        msgBox = QMessageBox(self.parent())
        msgBox.setText('Do you really want to delete this script?')
        msgBox.setStandardButtons(QMessageBox.Yes | QMessageBox.No)

        ret = msgBox.exec_()
        if ret == QMessageBox.Yes:
            # hide widget first
            self.parent().layout.removeWidget(self)
            # to save the order without the widget
            self.parent().saveBO()
            # then delete it
            self.deleteLater()
            # check if file should be delete or just button removed
            if deleteFile:
                os.remove(os.path.join(SCRIPTSPATH, '%s.py' % self.text()))
                self.parent().initAll()

    def renameButton(self):
        # get new name, dialogbox checks if exists
        files = [f[:-3] for f in os.listdir(SCRIPTSPATH) if f.endswith(".py")]
        newName, replace = fileDialogBox(
            self, 'New script name', 'Enter new script name. Name will be converted to valid file name.', files)
        # if there is a new Name, None if no name, or canceled
        if newName and newName != self.text():
            # rename file
            os.rename(os.path.join(SCRIPTSPATH, '%s.py' % self.text()), os.path.join(
                SCRIPTSPATH, '%s.py' % newName))
            # rename button
            if replace:
                self.deleteLater()
            else:
                self.setText(newName)
                # save order to data file
                self.parent().saveBO()

    def editButton(self):
        # get filepath and open in editor
        webbrowser.open(
            '%s' % os.path.join(SCRIPTSPATH, '%s.py' % self.text()))

    def runScript(self):
        # run modo script
        # could implement a command to let it run in the persistent
        # interpreter, it's already out there somewhere ;)
        lx.eval('@{%s}' % os.path.join(SCRIPTSPATH, '%s.py' % self.text()))

    def mouseMoveEvent(self, e):
        # move buttons only with right click, as it is annoying to accidentally
        # drag with a (wacom) tablet
        if e.buttons() != Qt.MiddleButton:
            return

        # following code is mainly copied from stackoverflow (with a few changes) ...
        # write the relative cursor position to mime data
        mimeData = QMimeData()
        # pass index of self as mimeData as I haven't found a better way to get
        # it in dropEvent yet
        mimeData.setText(str(self.parent().layout.indexOf(self)))
        # add my on data type, to distinguish between a button drop and a new
        # script drop
        mimeData.setData('modo/scriptShelfButton', QByteArray(self.text()))

        # let's make it fancy. we'll show a "ghost" of the button as we drag
        # grab the button to a pixmap
        pixmap = QPixmap.grabWidget(self)
        # below makes the pixmap half transparent
        painter = QPainter(pixmap)
        painter.setCompositionMode(painter.CompositionMode_DestinationIn)
        painter.fillRect(pixmap.rect(), QColor(0, 0, 0, 127))
        painter.end()

        # make a QDrag
        drag = QDrag(self)
        # put our MimeData
        drag.setMimeData(mimeData)
        # set its Pixmap
        drag.setPixmap(pixmap)
        # shift the Pixmap so that it coincides with the cursor position
        drag.setHotSpot(e.pos())

        # start the drag operation
        drag.exec_(Qt.MoveAction)
        # check if it is dragged outside the palette
        target = drag.target()
        print target
        print self
        if not target:
            # remove it
            self.removeButton()
        # ... until here


class scriptShelfWidget(QWidget):
    # we need to keep track of all instances to update all of them if needed
    _instances = []
    _toolbuttonstyles = {0:Qt.ToolButtonTextOnly, 1:Qt.ToolButtonTextUnderIcon, 2:Qt.ToolButtonIconOnly}

    def __init__(self, parent=None):
        self.__class__._instances.append(self)
        #print 'scriptShelfWidget init'
        # dunno what this does, but it's super ... well, actually some class
        # related stuff I don't understand yet or didn't bother to look into
        super(scriptShelfWidget, self).__init__()
        # set accept drops for drag and drop of scripts and buttons
        #self.iconDisp = Qt.ToolButtonTextOnly

        self.scriptShelfName = None

        self.layout = QBoxLayout(QBoxLayout.TopToBottom, self)
        self.layout.setContentsMargins(1, 1, 1, 1)

        self.initUI()

        self.setAcceptDrops(True)

        self.setContextMenuPolicy(Qt.ActionsContextMenu)
        # create and add delete menu item
        tIcon = QAction(self)
        tIcon.setText("Toggle Icons")
        tIcon.triggered.connect(self.toggleIcons)
        self.addAction(tIcon)
        change = QAction(self)
        change.setText("Change Shelf")
        change.triggered.connect(self.changeShelfName)
        self.addAction(change)
        add = QAction(self)
        add.setText("Add Shelf")
        add.triggered.connect(self.setShelfName)
        self.addAction(add)
        delete = QAction(self)
        delete.setText("Delete Shelf")
        delete.triggered.connect(self.deleteShelf)
        self.addAction(delete)

    def toggleIcons(self):
        if self.iconDisp == Qt.ToolButtonTextOnly:
            self.iconDisp = Qt.ToolButtonTextUnderIcon
        elif self.iconDisp == Qt.ToolButtonTextUnderIcon:
            self.iconDisp = Qt.ToolButtonIconOnly
        else:
            self.iconDisp = Qt.ToolButtonTextOnly
        print 'toggle'
        self.saveBO()
        self.initUI()

    def setShelfName(self, name = None):
        #print 'setShelfName'
        if name:
            self.scriptShelfName = name
        else:
            files = [f[:-4] for f in os.listdir(DATAPATH) if f.endswith(".txt")]
            text, replace = fileDialogBox(self, 'Shelf name', 'Enter new shelf name. Name will be converted to valid file name.', files)
            if text:
                x = os.path.join(DATAPATH, '%s.txt'%text)
                open(x, 'a').close()
                self.scriptShelfName = text
        self.initUI()

    def changeShelfName(self):
        files = [f[:-4] for f in os.listdir(DATAPATH) if f.endswith(".txt")]
        #print files
        name, ok= QInputDialog.getItem(self, 'Change Shelf', 'Please select shelf:', files, current=files.index(self.scriptShelfName), editable=False)
        if ok:
            self.scriptShelfName = name
            self.initUI()

    def deleteShelf(self):
        files = [f[:-4] for f in os.listdir(DATAPATH) if f.endswith(".txt") and f != 'main.txt']
        # get name of shelf to delete
        name, ok = QInputDialog.getItem(self, 'Delete Shelf', 'Please select shelf to delete:', files, current=files.index(self.scriptShelfName), editable=False)
        # check if ok
        if ok:
            # check if not main shelf
            if name != 'main':
                # check if the user is sure about this decision
                msgBox = QMessageBox(self.parent())
                msgBox.setText('Do you really want to delete this shelf "%s"?'%name)
                msgBox.setStandardButtons(QMessageBox.Yes | QMessageBox.No)

                ret = msgBox.exec_()
                if ret == QMessageBox.Yes:
                    print 'yes'
                    if self.scriptShelfName == name:
                        print 'back to main'
                        self.scriptShelfName = 'main'
                    # delete file
                    os.remove(os.path.join(DATAPATH, '%s.txt' %name))
                    self.initUI()
            elif name == 'main':
                msgBox = QMessageBox(self.parent())
                msgBox.setText('You cannot delete the main shelf!')
                msgBox.setStandardButtons(QMessageBox.Ok)
                ret = msgBox.exec_()

    def clearLayout(self, layout):
        if layout is not None:
            while layout.count():
                item = layout.takeAt(0)
                widget = item.widget()
                if widget is not None:
                    widget.deleteLater()
                """
                else:
                    self.clearLayout(item.layout())
                """
    @classmethod
    def initAll(cls):
        for inst in cls._instances:
            inst.initUI()

    def initUI(self):
        print 'initUI'
        if self.layout:
            self.clearLayout(self.layout)

        # set a layout, vertical has to do for now, I might add others later
        # read data, atm it's just the order of scripts on the shelf
        data, self.iconDisp = self.loadBO()
        # read python files on disc
        files = [f[:-3] for f in os.listdir(SCRIPTSPATH) if f.endswith(".py")]
        # check if there is an order set,
        if not data:
            # no data, could also be problems with file reading
            # list python files alphabetically
            for file in files:
                # create a custom button for each python file
                # custom button is needed for rightclick and D&D
                buttonN = RightClickScriptButton(file,self)
                # add button to layout
                self.layout.addWidget(buttonN)
        else:
            # there is data
            for d in data:
                # check if the file actually exists
                if d in files:
                    # same as a few lines above
                    buttonN = RightClickScriptButton(d,self)
                    self.layout.addWidget(buttonN)
                    # delete file from files list so we can ...
                    files = [x for x in files if x != d]
            # ... add files that aren't in the data list for whatever reason
            # this is just a precaution as user might copy files in the dir or
            # it's "this time of the year" again, moonhase, you know
            # whatever...
            if self.scriptShelfName == 'main':
                for f in files:
                    buttonN = RightClickScriptButton(f,self)
                    self.layout.addWidget(buttonN)

        # add a stretch so the buttons aren't spread out
        self.stretchWid = self.layout.addStretch(1)

        self.setLayout(self.layout)
        print 'initUI done'

    def resizeWid(self, size):
        # this class is called by the parent widget to change the layout to
        # horizontal/vertical based on size of widget
        sX, sY = size
        if sY < sX:
            self.layout.setDirection(QBoxLayout.LeftToRight)
        else:
            self.layout.setDirection(QBoxLayout.TopToBottom)

    def loadBO(self):
        print 'loadBO'
        # try to open data file and read it ...
        if self.scriptShelfName == None:
            self.scriptShelfName = 'main'

        try:
            #print self.scriptShelfName
            filepath = os.path.join(DATAPATH, '%s.txt'%self.scriptShelfName)
            print filepath
            #print filepath
            file = open(filepath, 'r')

            data = file.readlines()
            file.close()
            print data
            if len(data) == 1:
                print 1
                return data[0].replace('\n','').split(','), Qt.ToolButtonTextOnly
            elif len(data) == 2 and data[1]:
                print 2
                return data[0].replace('\n','').split(','), eval(data[1])
            else:
                print 'none'
                return None, Qt.ToolButtonTextOnly
        # ... or catch IO errors
        #except IOError, err:
        except err:
            print 'IOError', err
            return None, Qt.ToolButtonTextOnly

    def saveBO(self):
        print 'saveBO'
        # save data file to disk
        if self.scriptShelfName == None:
                self.scriptShelfName == 'main'
        data = ','.join([self.layout.itemAt(index).widget().text() for index in range(self.layout.count() - 1)])
        data = data + '\n%s'%str(self.iconDisp).replace('PySide.QtCore.','')
        print data
        filepath = os.path.join(DATAPATH, '%s.txt'%self.scriptShelfName)
        try:
            file = open(filepath, 'w')
            file.write(data)
            file.close()
        except IOError, err:
            lx.out('Couldn\'t write settings file.')
        print 'saveBO done'

    def dragEnterEvent(self, e):
        # we need to accept this so D&D works
        e.accept()

    def getIndexFrom(self, pos):
        # this was originally a fancy shelf function copied from github
        widget = self.childAt(pos)
        if widget:
            index = self.layout.indexOf(widget)
            if index >= 0:
                return index
        #return None

        # this was a failed attempt to catch a drop between buttons,

        closest = (QWIDGETSIZE_MAX, None)

        for index in range(self.layout.count()-1):
            widget = self.layout.itemAt(index).widget()
            widgetPos = widget.geometry().center()
            assert isinstance(pos, QPoint)
            distance = widgetPos.y()-pos.y()
            print distance
            if distance >= 0 and distance < closest[0]:
                closest = (distance, widget)


        if closest[1] is not None:
            return self.layout.indexOf(closest[1])

        #Right side drop
        return self.layout.count()-2


    def dropEvent(self, e):
        # MAGIC, no, not really, just some drop event
        # check if it is a button D&D ...
        files = [f[:-3] for f in os.listdir(SCRIPTSPATH) if f.endswith(".py")]

        # check if it is a button D&D ...
        if e.mimeData().hasFormat('modo/scriptShelfButton'):
            # old index is passed as text as I haven't found a more elegant way
            # yet to access the index here
            oldIndex = int(e.mimeData().text())
            # get the index of the button the dragged button is dropped on
            dropIndex = self.getIndexFrom(e.pos())
            print dropIndex
            # check if it is dropped on a button
            print self
            print e.source().parent()

            # check if it is from the same shelf ...
            if e.source().parent() == self:
                if dropIndex != None:
                    # get the button widget
                    wid = self.layout.itemAt(oldIndex).widget()
                    # remove it at its current position
                    self.layout.removeWidget(wid)
                    # insert it at its new position
                    self.layout.insertWidget(dropIndex, wid)
            # or not ...
            else:
                text = e.mimeData().data('modo/scriptShelfButton')

                print self
                print text
                print 'findChild', [c.text() for c in self.findChildren(RightClickScriptButton)]

                # don't know this line is still needed
                if not dropIndex:
                    dropIndex = self.layout.count() - 1

                # check if the button already exists on this form
                wids = self.findChildren(RightClickScriptButton)
                widsNames = [c.text() for c in wids]

                if text not in widsNames:
                    buttonN = RightClickScriptButton(str(text),self)
                    self.layout.insertWidget(dropIndex, buttonN)
                else:
                    wid = wids[widsNames.index(text)]
                    self.layout.removeWidget(wid)
                    self.layout.insertWidget(dropIndex, wid)

                print 'drop else'
        # ... or a script file from disk ...
        elif e.mimeData().hasUrls():
            # loop over files
            for i, url in enumerate(e.mimeData().urls()):
                # get file path
                path = url.toLocalFile()
                print path

                fExt = [str(f) for f in QImageReader.supportedImageFormats()]
                # check if file and pythonfile
                if os.path.isfile(path) and path.endswith('.py'):
                    # get filename and replace .py extension if there
                    filename = os.path.split(path)[1].replace('.py', '')
                    # get new filename, because this is simpler and I'm too
                    # lazy to check if it exists etc.
                    text, replace = fileDialogBox(
                        self, 'Script name', 'Enter new script name. Name will be converted to valid file name.', files, filename)
                    # if there is a name input
                    if text:
                        # copy file
                        shutil.copyfile(path, os.path.join(SCRIPTSPATH, '%s.py' % text))
                        # and create button if it's not a replacement
                        if not replace:
                            buttonN = RightClickScriptButton(text,self)
                            wPos = self.layout.count() - 1
                            self.layout.insertWidget(wPos, buttonN)
                elif os.path.splitext(path.lower())[1][1:] in fExt:
                    widget = self.childAt(e.pos())
                    if widget:
                        widget.changeIcon(path)
                        print 'icon dropped'
        # ... or a script D&D ...
        elif e.mimeData().hasFormat('text/plain'):
            # get script
            script = e.mimeData().text()
            # let the user type in a name for the script
            text, replace = fileDialogBox(
                self, 'Script name', 'Enter new script name. Name will be converted to valid file name.', files)
            if text:
                # create new script file
                filepath = os.path.join(SCRIPTSPATH, '%s.py' % text)
                file = open(filepath, 'w')
                file.write(script)
                file.close()
                # create new button for that file at the end of the shelf
                if not replace:
                    buttonN = RightClickScriptButton(text,self)
                    wPos = self.layout.count() - 1
                    self.layout.insertWidget(wPos, buttonN)
        # ... or someone is messing around
        else:

            if e.mimeData().hasFormat('application/x-mododragdata'):
                print e.mimeData().data('application/x-mododragdata')
            lx.out('not supported')
        self.saveBO()

# needed to subclass another widget to implement resizeEvent() and for the
# scrollbars


class containerWidget(QWidget):

    def __init__(self, parent=None):
        super(containerWidget, self).__init__()
        # content Widget
        self.widget = scriptShelfWidget()

        # Scroll Area Properties
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setWidget(self.widget)

        # Scroll Area Layer add
        vLayout = QVBoxLayout(self)
        vLayout.setContentsMargins(1, 1, 1, 1)
        vLayout.addWidget(scroll)
        self.setLayout(vLayout)

    def resizeEvent(self, e):
        sX, sY = e.size().toTuple()
        self.widget.resizeWid((sX, sY))


# original Code copied from here http://modo.sdk.thefoundry.co.uk/wiki/Persistent_Data
class scriptShelfPersistData(lxifc.Visitor):
    '''This is the persistent data visitor class. It's is responsible for walking
       the XML structure inside the top level atom. The top (outer) level atom,
       in this example the 'PersistTest' atom, is created when our 'PersistData'
       instance is configured (see the '__init__' function of 'PersistTestCmd'
       below).

    '''
    def vis_Evaluate(self):
        # need a Persistence service object
        per_S = lx.service.Persistence()
        # start a hash element of type 'MainHash'
        per_S.Start("pane", lx.symbol.i_PERSIST_HASH)

        # Start an atom element of type 'StringAtom' - note the strings you use
        # for both the hash and atom elements are arbitrary, you can use anything
        # you like.
        per_S.Start("scriptShelfNum", lx.symbol.i_PERSIST_ATOM)
        # add a value of type string to the atom
        per_S.AddValue (lx.symbol.sTYPE_STRING)
        # close the atom element
        self.atom_str = per_S.End()
        # need an attribute object to store the value for the atom
        self.attr_str = lx.object.Attributes(self.atom_str)

        # close the 'MainHash' hash element.
        self.hash_main = per_S.End()

# need a global reference to the persistent data object we'll be creating.
persist_data = None

class scriptShelf(lxifc.CustomView):

    def __init__(self):
        self.form = None
        self.shelfNum = 'shelfNum123'

        global persist_data

        # check to see if 'persist_data' already exists (we only want to create
        # it once per session), create and configure it if not. This also creates
        # the outermost hash of the XML fragment for our persistent data entry
        # in the user config file.
        if persist_data == None:
            persist_data = scriptShelfPersistData()
            lx.service.Persistence().Configure("ScriptShelf", persist_data)


    def setShelfNum(self, num):
        self.shelfNum = num

    def customview_Init(self, pane):
        #print 'custumview init'
        if pane == None:
            return False

        custPane = lx.object.CustomPane(pane)
        if custPane.test() == False:
            return False

        # get the parent object and convert it to a PySide QWidget
        parent = custPane.GetParent()
        parentWidget = lx.getQWidget(parent)

        # Check that it suceeds
        if parentWidget is not None:
            # create container widget for scroll and resize to work properly
            self.form = containerWidget(self)
            layout = QGridLayout()
            layout.setContentsMargins(1, 1, 1, 1)
            layout.addWidget(self.form)
            parentWidget.setLayout(layout)
            return True

        return False

    def customview_StoreState(self, pane):
        #print 'storestate'
        shelf = self.form.findChild(scriptShelfWidget)
        custpane = lx.object.CustomPane(pane)
        id = custpane.GetIdentifier()
        persist_data.hash_main.Insert(id)
        persist_data.atom_str.Append()
        persist_data.attr_str.SetString(0, str(shelf.scriptShelfName))
        #print shelf.scriptShelfName

    def customview_RestoreState(self, pane):
        #print 'restorestate'
        custpane = lx.object.CustomPane(pane)
        id = custpane.GetIdentifier()
        try:
            persist_data.hash_main.Lookup(id)
            shelfName = persist_data.attr_str.GetString(0)

            #print shelfName
        except LookupError, err:
            shelfName = 'main'

        shelf = self.form.findChild(scriptShelfWidget)
        shelf.setShelfName(shelfName)


lx.bless(scriptShelf, "scriptShelf")


